import os
import json
import torch
import argparse
from tqdm import tqdm
from datasets import load_dataset
from transformers import AutoTokenizer, AutoConfig, AutoModelForCausalLM
import sys
import random
import numpy as np

torch.manual_seed(42)
torch.cuda.manual_seed(42)
torch.cuda.manual_seed_all(42)
np.random.seed(42)
random.seed(42)
torch.backends.cudnn.deterministic = True
torch.backends.cudnn.benchmark = False

sys.path.append("/fs/nexus-scratch/hjae/ShadowKV")
from models.llama import LlamaForCausalLM
from data.metrics import normalize_answer, postprocess_pred

def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument("--model_path", type=str, default="nvidia/Llama-3.1-Nemotron-Nano-8B-v1")
    parser.add_argument("--token_budget", type=int, default=1024)
    parser.add_argument("--compression_enabled", action="store_true")
    parser.add_argument("--compression_threshold", type=int, default=128)
    parser.add_argument("--compression_ratio", type=float, default=0.5)
    parser.add_argument("--window_size", type=int, default=512) 
    parser.add_argument("--max_samples", type=int, default=100)
    parser.add_argument("--output_dir", type=str, default="results")
    return parser.parse_args()

def load_model_and_tokenizer(args):
    print(f"Loading model from {args.model_path}...")
    tokenizer = AutoTokenizer.from_pretrained(
        args.model_path,
        use_fast=False
    )
    config = AutoConfig.from_pretrained(args.model_path)
    
    model = LlamaForCausalLM.from_pretrained(
        args.model_path,
        config=config,
        device_map="auto",
        torch_dtype=torch.float16
    )

    model.shadowkv_init(
        window_size=args.window_size,
        max_tokens=args.token_budget,
        compress_ratio=args.compression_ratio if args.compression_enabled else 1.0,
        compress_threshold=args.compression_threshold,
    )

    return model, tokenizer

def format_prompt(passage, question):
    return f"""Read the following passage and answer the question. Provide only the final answer without explanation.

Passage: {passage}

Question: {question}

Answer:"""

def extract_answer(response):
  
    if "Answer:" in response:
        answer_part = response.split("Answer:")[-1].strip()
        
        import re
        numbers = re.findall(r'\d+(?:\.\d+)?%?', answer_part)
        
        if numbers:
            return numbers[0]
        
        lines = answer_part.split("\n")
        if lines:
            first_line = lines[0].strip()
        
            words = first_line.split()
            for word in words:
                word_clean = word.strip(".,!?")
                if word_clean and word_clean.lower() not in ["the", "so", "answer", "is", "are", "was", "were", "will", "would", "could", "should"]:
                    return word_clean
    
    return "No answer found"

def is_correct(pred, golds):
    pred_norm = normalize_answer(postprocess_pred(pred))
    return any(pred_norm == normalize_answer(gold) for gold in golds)

def load_partial_results(output_file):
    if os.path.exists(output_file):
        with open(output_file, "r") as f:
            data = json.load(f)
            return data.get("results", []), data.get("correct", 0), data.get("total", 0)
    return [], 0, 0

def evaluate_drop(model, tokenizer, dataset, args, output_file):
    results, correct, total = load_partial_results(output_file)
    start_idx = total
    n_total = min(len(dataset), args.max_samples) if args.max_samples else len(dataset)
    batch_size = 100
    save_every = 50

    while start_idx < n_total:
        end_idx = min(start_idx + batch_size, n_total)
        subset = dataset.select(range(start_idx, end_idx))
        
        for sample in tqdm(subset, initial=start_idx, total=n_total):
            passage = sample["passage"]
            question = sample["question"]
            golds = sample["answers_spans"]["spans"]

            prompt = format_prompt(passage, question)
            inputs = tokenizer(prompt, return_tensors="pt").to(model.device)

            with torch.no_grad():
                outputs = model.generate(
                    inputs.input_ids,
                    max_new_tokens=64,  
                    temperature=0.1,  
                    top_p=0.9,
                    do_sample=True,
                    pad_token_id=tokenizer.eos_token_id,
                    eos_token_id=tokenizer.eos_token_id
                )

            response = tokenizer.decode(outputs[0], skip_special_tokens=True)
            predicted_answer = extract_answer(response)

            correct_flag = is_correct(predicted_answer, golds)
            if correct_flag:
                correct += 1
            total += 1

            results.append({
                "passage": passage,
                "question": question,
                "gold_answers": golds,
                "predicted_answer": predicted_answer,
                "is_correct": correct_flag,
                "response": response
            })

            if (total % save_every == 0) or (total == n_total):
                with open(output_file, "w") as f:
                    json.dump({
                        "args": vars(args),
                        "accuracy": correct / total if total > 0 else 0,
                        "results": results,
                        "correct": correct,
                        "total": total
                    }, f, indent=2)
                print(f"[Checkpoint] Saved {total} results to {output_file}")
        
        start_idx = end_idx
    
    return results, correct / total if total > 0 else 0

def main():
    args = parse_args()
    os.makedirs(args.output_dir, exist_ok=True)

    model, tokenizer = load_model_and_tokenizer(args)

    print("Loading DROP dataset...")
    dataset = load_dataset("ucinlp/drop")
    test_set = dataset["validation"]

    if args.max_samples is not None:
        test_set = test_set.shuffle(seed=42).select(range(min(args.max_samples, len(test_set))))
        print(f"Using {len(test_set)} samples for evaluation (randomly selected with seed=42)")

    print("Starting evaluation...")
    output_file = os.path.join(args.output_dir, "drop_results_shadowkv_nvidia_fixed.json")
    results, accuracy = evaluate_drop(model, tokenizer, test_set, args, output_file)

    print(f"\nFinal accuracy: {accuracy:.2%}")
    print(f"Results saved to {output_file}")

if __name__ == "__main__":
    main()
